import { Fragment, computed, createBlock, createCommentVNode, createElementBlock, createElementVNode, createTextVNode, createVNode, defineComponent, isVNode, mergeProps, nextTick, normalizeClass, normalizeProps, normalizeStyle, onBeforeUnmount, onMounted, openBlock, ref, renderList, renderSlot, resolveDynamicComponent, toDisplayString, unref, useAttrs, watch, watchEffect, withCtx } from "vue";

//#region src/packages/state.ts
let toastsCounter = 1;
var Observer = class {
	subscribers;
	toasts;
	dismissedToasts;
	constructor() {
		this.subscribers = [];
		this.toasts = [];
		this.dismissedToasts = new Set();
	}
	subscribe = (subscriber) => {
		this.subscribers.push(subscriber);
		return () => {
			const index = this.subscribers.indexOf(subscriber);
			this.subscribers.splice(index, 1);
		};
	};
	publish = (data) => {
		this.subscribers.forEach((subscriber) => subscriber(data));
	};
	addToast = (data) => {
		this.publish(data);
		this.toasts = [...this.toasts, data];
	};
	create = (data) => {
		const { message,...rest } = data;
		const id = typeof data.id === "number" || data.id && data.id?.length > 0 ? data.id : toastsCounter++;
		const alreadyExists = this.toasts.find((toast$1) => {
			return toast$1.id === id;
		});
		const dismissible = data.dismissible === void 0 ? true : data.dismissible;
		if (this.dismissedToasts.has(id)) this.dismissedToasts.delete(id);
		if (alreadyExists) this.toasts = this.toasts.map((toast$1) => {
			if (toast$1.id === id) {
				this.publish({
					...toast$1,
					...data,
					id,
					title: message
				});
				return {
					...toast$1,
					...data,
					id,
					dismissible,
					title: message
				};
			}
			return toast$1;
		});
		else this.addToast({
			title: message,
			...rest,
			dismissible,
			id
		});
		return id;
	};
	dismiss = (id) => {
		if (id) {
			this.dismissedToasts.add(id);
			requestAnimationFrame(() => this.subscribers.forEach((subscriber) => subscriber({
				id,
				dismiss: true
			})));
		} else this.toasts.forEach((toast$1) => {
			this.subscribers.forEach((subscriber) => subscriber({
				id: toast$1.id,
				dismiss: true
			}));
		});
		return id;
	};
	message = (message, data) => {
		return this.create({
			...data,
			message,
			type: "default"
		});
	};
	error = (message, data) => {
		return this.create({
			...data,
			type: "error",
			message
		});
	};
	success = (message, data) => {
		return this.create({
			...data,
			type: "success",
			message
		});
	};
	info = (message, data) => {
		return this.create({
			...data,
			type: "info",
			message
		});
	};
	warning = (message, data) => {
		return this.create({
			...data,
			type: "warning",
			message
		});
	};
	loading = (message, data) => {
		return this.create({
			...data,
			type: "loading",
			message
		});
	};
	promise = (promise, data) => {
		if (!data) return;
		let id;
		if (data.loading !== void 0) id = this.create({
			...data,
			promise,
			type: "loading",
			message: data.loading,
			description: typeof data.description !== "function" ? data.description : void 0
		});
		const p = Promise.resolve(promise instanceof Function ? promise() : promise);
		let shouldDismiss = id !== void 0;
		let result;
		const originalPromise = p.then(async (response) => {
			result = ["resolve", response];
			const isVueComponent = isVNode(response);
			if (isVueComponent) {
				shouldDismiss = false;
				this.create({
					id,
					type: "default",
					message: response
				});
			} else if (isHttpResponse(response) && !response.ok) {
				shouldDismiss = false;
				const promiseData = typeof data.error === "function" ? await data.error(`HTTP error! status: ${response.status}`) : data.error;
				const description = typeof data.description === "function" ? await data.description(`HTTP error! status: ${response.status}`) : data.description;
				const isExtendedResult = typeof promiseData === "object" && !isVNode(promiseData);
				const toastSettings = isExtendedResult ? promiseData : {
					message: promiseData || "",
					id: id || ""
				};
				this.create({
					id,
					type: "error",
					description,
					...toastSettings
				});
			} else if (response instanceof Error) {
				shouldDismiss = false;
				const promiseData = typeof data.error === "function" ? await data.error(response) : data.error;
				const description = typeof data.description === "function" ? await data.description(response) : data.description;
				const isExtendedResult = typeof promiseData === "object" && !isVNode(promiseData);
				const toastSettings = isExtendedResult ? promiseData : {
					message: promiseData || "",
					id: id || ""
				};
				this.create({
					id,
					type: "error",
					description,
					...toastSettings
				});
			} else if (data.success !== void 0) {
				shouldDismiss = false;
				const promiseData = typeof data.success === "function" ? await data.success(response) : data.success;
				const description = typeof data.description === "function" ? await data.description(response) : data.description;
				const isExtendedResult = typeof promiseData === "object" && !isVNode(promiseData);
				const toastSettings = isExtendedResult ? promiseData : {
					message: promiseData || "",
					id: id || ""
				};
				this.create({
					id,
					type: "success",
					description,
					...toastSettings
				});
			}
		}).catch(async (error) => {
			result = ["reject", error];
			if (data.error !== void 0) {
				shouldDismiss = false;
				const promiseData = typeof data.error === "function" ? await data.error(error) : data.error;
				const description = typeof data.description === "function" ? await data.description(error) : data.description;
				const isExtendedResult = typeof promiseData === "object" && !isVNode(promiseData);
				const toastSettings = isExtendedResult ? promiseData : {
					message: promiseData || "",
					id: id || ""
				};
				this.create({
					id,
					type: "error",
					description,
					...toastSettings
				});
			}
		}).finally(() => {
			if (shouldDismiss) {
				this.dismiss(id);
				id = void 0;
			}
			data.finally?.();
		});
		const unwrap = () => new Promise((resolve, reject) => originalPromise.then(() => result[0] === "reject" ? reject(result[1]) : resolve(result[1])).catch(reject));
		if (typeof id !== "string" && typeof id !== "number") return { unwrap };
		else return Object.assign(id, { unwrap });
	};
	custom = (component, data) => {
		const id = data?.id || toastsCounter++;
		const alreadyExists = this.toasts.find((toast$1) => {
			return toast$1.id === id;
		});
		const dismissible = data?.dismissible === void 0 ? true : data.dismissible;
		if (this.dismissedToasts.has(id)) this.dismissedToasts.delete(id);
		if (alreadyExists) this.toasts = this.toasts.map((toast$1) => {
			if (toast$1.id === id) {
				this.publish({
					...toast$1,
					component,
					dismissible,
					id,
					...data
				});
				return {
					...toast$1,
					component,
					dismissible,
					id,
					...data
				};
			}
			return toast$1;
		});
		else this.addToast({
			component,
			dismissible,
			id,
			...data
		});
		return id;
	};
	getActiveToasts = () => {
		return this.toasts.filter((toast$1) => !this.dismissedToasts.has(toast$1.id));
	};
};
const ToastState = new Observer();
function toastFunction(message, data) {
	const id = data?.id || toastsCounter++;
	ToastState.create({
		message,
		id,
		type: "default",
		...data
	});
	return id;
}
const isHttpResponse = (data) => {
	return data && typeof data === "object" && "ok" in data && typeof data.ok === "boolean" && "status" in data && typeof data.status === "number";
};
const basicToast = toastFunction;
const getHistory = () => ToastState.toasts;
const getToasts = () => ToastState.getActiveToasts();
const toast = Object.assign(basicToast, {
	success: ToastState.success,
	info: ToastState.info,
	warning: ToastState.warning,
	error: ToastState.error,
	custom: ToastState.custom,
	message: ToastState.message,
	promise: ToastState.promise,
	dismiss: ToastState.dismiss,
	loading: ToastState.loading
}, {
	getHistory,
	getToasts
});

//#endregion
//#region src/packages/types.ts
function isAction(action) {
	return action.label !== void 0;
}

//#endregion
//#region src/packages/constant.ts
const VISIBLE_TOASTS_AMOUNT = 3;
const VIEWPORT_OFFSET = "24px";
const MOBILE_VIEWPORT_OFFSET = "16px";
const TOAST_LIFETIME = 4e3;
const TOAST_WIDTH = 356;
const GAP = 14;
const SWIPE_THRESHOLD = 45;
const TIME_BEFORE_UNMOUNT = 200;

//#endregion
//#region src/packages/hooks.ts
function useIsDocumentHidden() {
	const isDocumentHidden = ref(false);
	watchEffect(() => {
		const callback = () => {
			isDocumentHidden.value = document.hidden;
		};
		document.addEventListener("visibilitychange", callback);
		return () => window.removeEventListener("visibilitychange", callback);
	});
	return { isDocumentHidden };
}
function cn(...classes) {
	return classes.filter(Boolean).join(" ");
}
function getDefaultSwipeDirections(position) {
	const [y, x] = position.split("-");
	const directions = [];
	if (y) directions.push(y);
	if (x) directions.push(x);
	return directions;
}
function assignOffset(defaultOffset, mobileOffset) {
	const styles = {};
	[defaultOffset, mobileOffset].forEach((offset, index) => {
		const isMobile = index === 1;
		const prefix = isMobile ? "--mobile-offset" : "--offset";
		const defaultValue = isMobile ? MOBILE_VIEWPORT_OFFSET : VIEWPORT_OFFSET;
		function assignAll(offset$1) {
			[
				"top",
				"right",
				"bottom",
				"left"
			].forEach((key) => {
				styles[`${prefix}-${key}`] = typeof offset$1 === "number" ? `${offset$1}px` : offset$1;
			});
		}
		if (typeof offset === "number" || typeof offset === "string") assignAll(offset);
		else if (typeof offset === "object") [
			"top",
			"right",
			"bottom",
			"left"
		].forEach((key) => {
			if (offset[key] === void 0) styles[`${prefix}-${key}`] = defaultValue;
			else styles[`${prefix}-${key}`] = typeof offset[key] === "number" ? `${offset[key]}px` : offset[key];
		});
		else assignAll(defaultValue);
	});
	return styles;
}
function useVueSonner() {
	const activeToasts = ref([]);
	watchEffect((onInvalidate) => {
		const unsubscribe = ToastState.subscribe((toast$1) => {
			if ("dismiss" in toast$1 && toast$1.dismiss) {
				activeToasts.value = activeToasts.value.filter((t) => t.id !== toast$1.id);
				return;
			}
			nextTick(() => {
				const existingToastIndex = activeToasts.value.findIndex((t) => t.id === toast$1.id);
				if (existingToastIndex !== -1) {
					const updatedToasts = [...activeToasts.value];
					updatedToasts[existingToastIndex] = {
						...updatedToasts[existingToastIndex],
						...toast$1
					};
					activeToasts.value = updatedToasts;
				} else activeToasts.value = [toast$1, ...activeToasts.value];
			});
		});
		onInvalidate(() => {
			unsubscribe();
		});
	});
	return { activeToasts };
}

//#endregion
//#region src/packages/Toast.vue?vue&type=script&setup=true&lang.ts
const _hoisted_1$7 = [
	"data-rich-colors",
	"data-styled",
	"data-mounted",
	"data-promise",
	"data-swiped",
	"data-removed",
	"data-visible",
	"data-y-position",
	"data-x-position",
	"data-index",
	"data-front",
	"data-swiping",
	"data-dismissible",
	"data-type",
	"data-invert",
	"data-swipe-out",
	"data-swipe-direction",
	"data-expanded",
	"data-testid"
];
const _hoisted_2$2 = [
	"aria-label",
	"data-disabled",
	"data-close-button-position"
];
var Toast_vue_vue_type_script_setup_true_lang_default = /* @__PURE__ */ defineComponent({
	__name: "Toast",
	props: {
		toast: {},
		toasts: {},
		index: {},
		swipeDirections: {},
		expanded: { type: Boolean },
		invert: { type: Boolean },
		heights: {},
		gap: {},
		position: {},
		closeButtonPosition: {},
		visibleToasts: {},
		expandByDefault: { type: Boolean },
		closeButton: { type: Boolean },
		interacting: { type: Boolean },
		style: {},
		cancelButtonStyle: {},
		actionButtonStyle: {},
		duration: {},
		class: {},
		unstyled: { type: Boolean },
		descriptionClass: {},
		loadingIcon: {},
		classes: {},
		icons: {},
		closeButtonAriaLabel: {},
		defaultRichColors: { type: Boolean }
	},
	emits: [
		"update:heights",
		"update:height",
		"removeToast"
	],
	setup(__props, { emit: __emit }) {
		const props = __props;
		const emit = __emit;
		const swipeDirection = ref(null);
		const swipeOutDirection = ref(null);
		const mounted = ref(false);
		const removed = ref(false);
		const swiping = ref(false);
		const swipeOut = ref(false);
		const swiped = ref(false);
		const offsetBeforeRemove = ref(0);
		const initialHeight = ref(0);
		const remainingTime = ref(props.toast.duration || props.duration || TOAST_LIFETIME);
		const dragStartTime = ref(null);
		const toastRef = ref(null);
		const isFront = computed(() => props.index === 0);
		const isVisible = computed(() => props.index + 1 <= props.visibleToasts);
		const toastType = computed(() => props.toast.type);
		const dismissible = computed(() => props.toast.dismissible !== false);
		const toastClass = computed(() => props.toast.class || "");
		const toastDescriptionClass = computed(() => props.descriptionClass || "");
		const heightIndex = computed(() => {
			const currentPosition = props.toast.position || props.position;
			const samePositionHeights = props.heights.filter((h) => h.position === currentPosition);
			const index = samePositionHeights.findIndex((height) => height.toastId === props.toast.id);
			return index >= 0 ? index : 0;
		});
		const toastsHeightBefore = computed(() => {
			const currentPosition = props.toast.position || props.position;
			const samePositionHeights = props.heights.filter((h) => h.position === currentPosition);
			return samePositionHeights.reduce((prev, curr, reducerIndex) => {
				if (reducerIndex >= heightIndex.value) return prev;
				return prev + curr.height;
			}, 0);
		});
		const offset = computed(() => heightIndex.value * props.gap + toastsHeightBefore.value || 0);
		const closeButton = computed(() => props.toast.closeButton ?? props.closeButton);
		const duration = computed(() => props.toast.duration || props.duration || TOAST_LIFETIME);
		const closeTimerStartTimeRef = ref(0);
		const lastCloseTimerStartTimeRef = ref(0);
		const pointerStartRef = ref(null);
		const coords = computed(() => props.position.split("-"));
		const y = computed(() => coords.value[0]);
		const x = computed(() => coords.value[1]);
		const isStringOfTitle = computed(() => typeof props.toast.title !== "string");
		const isStringOfDescription = computed(() => typeof props.toast.description !== "string");
		const { isDocumentHidden } = useIsDocumentHidden();
		const disabled = computed(() => toastType.value && toastType.value === "loading");
		onMounted(() => {
			mounted.value = true;
			remainingTime.value = duration.value;
		});
		watchEffect(async () => {
			if (!mounted.value || !toastRef.value) return;
			await nextTick();
			const toastNode = toastRef.value;
			const originalHeight = toastNode.style.height;
			toastNode.style.height = "auto";
			const newHeight = toastNode.getBoundingClientRect().height;
			toastNode.style.height = originalHeight;
			initialHeight.value = newHeight;
			emit("update:height", {
				toastId: props.toast.id,
				height: newHeight,
				position: props.toast.position || props.position
			});
		});
		function deleteToast() {
			removed.value = true;
			offsetBeforeRemove.value = offset.value;
			setTimeout(() => {
				emit("removeToast", props.toast);
			}, TIME_BEFORE_UNMOUNT);
		}
		function handleCloseToast() {
			if (disabled.value || !dismissible.value) return {};
			deleteToast();
			props.toast.onDismiss?.(props.toast);
		}
		function onPointerDown(event) {
			if (event.button === 2) return;
			if (disabled.value || !dismissible.value) return;
			dragStartTime.value = new Date();
			offsetBeforeRemove.value = offset.value;
			event.target.setPointerCapture(event.pointerId);
			if (event.target.tagName === "BUTTON") return;
			swiping.value = true;
			pointerStartRef.value = {
				x: event.clientX,
				y: event.clientY
			};
		}
		function onPointerUp() {
			if (swipeOut.value || !dismissible.value) return;
			pointerStartRef.value = null;
			const swipeAmountX = Number(toastRef.value?.style.getPropertyValue("--swipe-amount-x").replace("px", "") || 0);
			const swipeAmountY = Number(toastRef.value?.style.getPropertyValue("--swipe-amount-y").replace("px", "") || 0);
			const timeTaken = new Date().getTime() - (dragStartTime.value?.getTime() || 0);
			const swipeAmount = swipeDirection.value === "x" ? swipeAmountX : swipeAmountY;
			const velocity = Math.abs(swipeAmount) / timeTaken;
			if (Math.abs(swipeAmount) >= SWIPE_THRESHOLD || velocity > .11) {
				offsetBeforeRemove.value = offset.value;
				props.toast.onDismiss?.(props.toast);
				if (swipeDirection.value === "x") swipeOutDirection.value = swipeAmountX > 0 ? "right" : "left";
				else swipeOutDirection.value = swipeAmountY > 0 ? "down" : "up";
				deleteToast();
				swipeOut.value = true;
				return;
			} else {
				toastRef.value?.style.setProperty("--swipe-amount-x", `0px`);
				toastRef.value?.style.setProperty("--swipe-amount-y", `0px`);
			}
			swiped.value = false;
			swiping.value = false;
			swipeDirection.value = null;
		}
		function onPointerMove(event) {
			if (!pointerStartRef.value || !dismissible.value) return;
			const isHighlighted = window?.getSelection()?.toString()?.length ?? false;
			if (isHighlighted) return;
			const yDelta = event.clientY - pointerStartRef.value.y;
			const xDelta = event.clientX - pointerStartRef.value.x;
			const swipeDirections = props.swipeDirections ?? getDefaultSwipeDirections(props.position);
			if (!swipeDirection.value && (Math.abs(xDelta) > 1 || Math.abs(yDelta) > 1)) swipeDirection.value = Math.abs(xDelta) > Math.abs(yDelta) ? "x" : "y";
			let swipeAmount = {
				x: 0,
				y: 0
			};
			const getDampening = (delta) => {
				const factor = Math.abs(delta) / 20;
				return 1 / (1.5 + factor);
			};
			if (swipeDirection.value === "y") {
				if (swipeDirections.includes("top") || swipeDirections.includes("bottom")) if (swipeDirections.includes("top") && yDelta < 0 || swipeDirections.includes("bottom") && yDelta > 0) swipeAmount.y = yDelta;
				else {
					const dampenedDelta = yDelta * getDampening(yDelta);
					swipeAmount.y = Math.abs(dampenedDelta) < Math.abs(yDelta) ? dampenedDelta : yDelta;
				}
			} else if (swipeDirection.value === "x") {
				if (swipeDirections.includes("left") || swipeDirections.includes("right")) if (swipeDirections.includes("left") && xDelta < 0 || swipeDirections.includes("right") && xDelta > 0) swipeAmount.x = xDelta;
				else {
					const dampenedDelta = xDelta * getDampening(xDelta);
					swipeAmount.x = Math.abs(dampenedDelta) < Math.abs(xDelta) ? dampenedDelta : xDelta;
				}
			}
			if (Math.abs(swipeAmount.x) > 0 || Math.abs(swipeAmount.y) > 0) swiped.value = true;
			toastRef.value?.style.setProperty("--swipe-amount-x", `${swipeAmount.x}px`);
			toastRef.value?.style.setProperty("--swipe-amount-y", `${swipeAmount.y}px`);
		}
		onMounted(() => {
			mounted.value = true;
			if (!toastRef.value) return;
			const height = toastRef.value.getBoundingClientRect().height;
			initialHeight.value = height;
			const newHeights = [{
				toastId: props.toast.id,
				height,
				position: props.toast.position
			}, ...props.heights];
			emit("update:heights", newHeights);
		});
		onBeforeUnmount(() => {
			if (toastRef.value) emit("removeToast", props.toast);
		});
		watchEffect((onInvalidate) => {
			if (props.toast.promise && toastType.value === "loading" || props.toast.duration === Infinity || props.toast.type === "loading") return;
			let timeoutId;
			const pauseTimer = () => {
				if (lastCloseTimerStartTimeRef.value < closeTimerStartTimeRef.value) {
					const elapsedTime = new Date().getTime() - closeTimerStartTimeRef.value;
					remainingTime.value = remainingTime.value - elapsedTime;
				}
				lastCloseTimerStartTimeRef.value = new Date().getTime();
			};
			const startTimer = () => {
				if (remainingTime.value === Infinity) return;
				closeTimerStartTimeRef.value = new Date().getTime();
				timeoutId = setTimeout(() => {
					props.toast.onAutoClose?.(props.toast);
					deleteToast();
				}, remainingTime.value);
			};
			if (props.expanded || props.interacting || isDocumentHidden.value) pauseTimer();
			else startTimer();
			onInvalidate(() => {
				clearTimeout(timeoutId);
			});
		});
		watch(() => props.toast.delete, (value) => {
			if (value !== void 0 && value) {
				deleteToast();
				props.toast.onDismiss?.(props.toast);
			}
		}, { deep: true });
		function handleDragEnd() {
			swiping.value = false;
			swipeDirection.value = null;
			pointerStartRef.value = null;
		}
		return (_ctx, _cache) => {
			return openBlock(), createElementBlock("li", {
				tabindex: "0",
				ref_key: "toastRef",
				ref: toastRef,
				class: normalizeClass(unref(cn)(props.class, toastClass.value, _ctx.classes?.toast, _ctx.toast.classes?.toast, _ctx.classes?.[toastType.value], _ctx.toast?.classes?.[toastType.value])),
				"data-sonner-toast": "",
				"data-rich-colors": _ctx.toast.richColors ?? _ctx.defaultRichColors,
				"data-styled": !Boolean(_ctx.toast.component || _ctx.toast?.unstyled || _ctx.unstyled),
				"data-mounted": mounted.value,
				"data-promise": Boolean(_ctx.toast.promise),
				"data-swiped": swiped.value,
				"data-removed": removed.value,
				"data-visible": isVisible.value,
				"data-y-position": y.value,
				"data-x-position": x.value,
				"data-index": _ctx.index,
				"data-front": isFront.value,
				"data-swiping": swiping.value,
				"data-dismissible": dismissible.value,
				"data-type": toastType.value,
				"data-invert": _ctx.toast.invert || _ctx.invert,
				"data-swipe-out": swipeOut.value,
				"data-swipe-direction": swipeOutDirection.value,
				"data-expanded": Boolean(_ctx.expanded || _ctx.expandByDefault && mounted.value),
				"data-testid": _ctx.toast.testId,
				style: normalizeStyle({
					"--index": _ctx.index,
					"--toasts-before": _ctx.index,
					"--z-index": _ctx.toasts.length - _ctx.index,
					"--offset": `${removed.value ? offsetBeforeRemove.value : offset.value}px`,
					"--initial-height": _ctx.expandByDefault ? "auto" : `${initialHeight.value}px`,
					..._ctx.style,
					...props.toast.style
				}),
				onDragend: handleDragEnd,
				onPointerdown: onPointerDown,
				onPointerup: onPointerUp,
				onPointermove: onPointerMove
			}, [closeButton.value && !_ctx.toast.component && toastType.value !== "loading" ? (openBlock(), createElementBlock("button", {
				key: 0,
				"aria-label": _ctx.closeButtonAriaLabel || "Close toast",
				"data-disabled": disabled.value,
				"data-close-button": "true",
				"data-close-button-position": _ctx.closeButtonPosition,
				class: normalizeClass(unref(cn)(_ctx.classes?.closeButton, _ctx.toast?.classes?.closeButton)),
				onClick: handleCloseToast
			}, [_ctx.icons?.close ? (openBlock(), createBlock(resolveDynamicComponent(_ctx.icons?.close), { key: 0 })) : renderSlot(_ctx.$slots, "close-icon", { key: 1 })], 10, _hoisted_2$2)) : createCommentVNode("v-if", true), _ctx.toast.component ? (openBlock(), createBlock(resolveDynamicComponent(_ctx.toast.component), mergeProps({ key: 1 }, _ctx.toast.componentProps, {
				onCloseToast: handleCloseToast,
				isPaused: _ctx.$props.expanded || _ctx.$props.interacting || unref(isDocumentHidden)
			}), null, 16, ["isPaused"])) : (openBlock(), createElementBlock(Fragment, { key: 2 }, [
				toastType.value !== "default" || _ctx.toast.icon || _ctx.toast.promise ? (openBlock(), createElementBlock("div", {
					key: 0,
					"data-icon": "",
					class: normalizeClass(unref(cn)(_ctx.classes?.icon, _ctx.toast?.classes?.icon))
				}, [_ctx.toast.icon ? (openBlock(), createBlock(resolveDynamicComponent(_ctx.toast.icon), { key: 0 })) : (openBlock(), createElementBlock(Fragment, { key: 1 }, [toastType.value === "loading" ? renderSlot(_ctx.$slots, "loading-icon", { key: 0 }) : toastType.value === "success" ? renderSlot(_ctx.$slots, "success-icon", { key: 1 }) : toastType.value === "error" ? renderSlot(_ctx.$slots, "error-icon", { key: 2 }) : toastType.value === "warning" ? renderSlot(_ctx.$slots, "warning-icon", { key: 3 }) : toastType.value === "info" ? renderSlot(_ctx.$slots, "info-icon", { key: 4 }) : createCommentVNode("v-if", true)], 64))], 2)) : createCommentVNode("v-if", true),
				createElementVNode("div", {
					"data-content": "",
					class: normalizeClass(unref(cn)(_ctx.classes?.content, _ctx.toast?.classes?.content))
				}, [createElementVNode("div", {
					"data-title": "",
					class: normalizeClass(unref(cn)(_ctx.classes?.title, _ctx.toast.classes?.title))
				}, [isStringOfTitle.value ? (openBlock(), createBlock(resolveDynamicComponent(_ctx.toast.title), normalizeProps(mergeProps({ key: 0 }, _ctx.toast.componentProps)), null, 16)) : (openBlock(), createElementBlock(Fragment, { key: 1 }, [createTextVNode(toDisplayString(_ctx.toast.title), 1)], 64))], 2), _ctx.toast.description ? (openBlock(), createElementBlock("div", {
					key: 0,
					"data-description": "",
					class: normalizeClass(unref(cn)(_ctx.descriptionClass, toastDescriptionClass.value, _ctx.classes?.description, _ctx.toast.classes?.description))
				}, [isStringOfDescription.value ? (openBlock(), createBlock(resolveDynamicComponent(_ctx.toast.description), normalizeProps(mergeProps({ key: 0 }, _ctx.toast.componentProps)), null, 16)) : (openBlock(), createElementBlock(Fragment, { key: 1 }, [createTextVNode(toDisplayString(_ctx.toast.description), 1)], 64))], 2)) : createCommentVNode("v-if", true)], 2),
				_ctx.toast.cancel ? (openBlock(), createElementBlock("button", {
					key: 1,
					style: normalizeStyle(_ctx.toast.cancelButtonStyle || _ctx.cancelButtonStyle),
					class: normalizeClass(unref(cn)(_ctx.classes?.cancelButton, _ctx.toast.classes?.cancelButton)),
					"data-button": "",
					"data-cancel": "",
					onClick: _cache[0] || (_cache[0] = (event) => {
						if (!unref(isAction)(_ctx.toast.cancel)) return;
						if (!dismissible.value) return;
						_ctx.toast.cancel.onClick?.(event);
						deleteToast();
					})
				}, toDisplayString(unref(isAction)(_ctx.toast.cancel) ? _ctx.toast.cancel?.label : _ctx.toast.cancel), 7)) : createCommentVNode("v-if", true),
				_ctx.toast.action ? (openBlock(), createElementBlock("button", {
					key: 2,
					style: normalizeStyle(_ctx.toast.actionButtonStyle || _ctx.actionButtonStyle),
					class: normalizeClass(unref(cn)(_ctx.classes?.actionButton, _ctx.toast.classes?.actionButton)),
					"data-button": "",
					"data-action": "",
					onClick: _cache[1] || (_cache[1] = (event) => {
						if (!unref(isAction)(_ctx.toast.action)) return;
						_ctx.toast.action.onClick?.(event);
						if (event.defaultPrevented) return;
						deleteToast();
					})
				}, toDisplayString(unref(isAction)(_ctx.toast.action) ? _ctx.toast.action?.label : _ctx.toast.action), 7)) : createCommentVNode("v-if", true)
			], 64))], 46, _hoisted_1$7);
		};
	}
});

//#endregion
//#region src/packages/Toast.vue
var Toast_default = Toast_vue_vue_type_script_setup_true_lang_default;

//#endregion
//#region \0/plugin-vue/export-helper
var export_helper_default = (sfc, props) => {
	const target = sfc.__vccOpts || sfc;
	for (const [key, val] of props) target[key] = val;
	return target;
};

//#endregion
//#region src/packages/assets/CloseIcon.vue
const _sfc_main$4 = {};
const _hoisted_1$6 = {
	xmlns: "http://www.w3.org/2000/svg",
	width: "12",
	height: "12",
	viewBox: "0 0 24 24",
	fill: "none",
	stroke: "currentColor",
	"stoke-width": "1.5",
	"stroke-linecap": "round",
	"stroke-linejoin": "round"
};
function _sfc_render$4(_ctx, _cache) {
	return openBlock(), createElementBlock("svg", _hoisted_1$6, _cache[0] || (_cache[0] = [createElementVNode("line", {
		x1: "18",
		y1: "6",
		x2: "6",
		y2: "18"
	}, null, -1), createElementVNode("line", {
		x1: "6",
		y1: "6",
		x2: "18",
		y2: "18"
	}, null, -1)]));
}
var CloseIcon_default = /* @__PURE__ */ export_helper_default(_sfc_main$4, [["render", _sfc_render$4]]);

//#endregion
//#region src/packages/assets/Loader.vue?vue&type=script&setup=true&lang.ts
const _hoisted_1$5 = ["data-visible"];
const _hoisted_2$1 = { class: "sonner-spinner" };
var Loader_vue_vue_type_script_setup_true_lang_default = /* @__PURE__ */ defineComponent({
	__name: "Loader",
	props: { visible: { type: Boolean } },
	setup(__props) {
		const bars = Array(12).fill(0);
		return (_ctx, _cache) => {
			return openBlock(), createElementBlock("div", {
				class: "sonner-loading-wrapper",
				"data-visible": _ctx.visible
			}, [createElementVNode("div", _hoisted_2$1, [(openBlock(true), createElementBlock(Fragment, null, renderList(unref(bars), (bar) => {
				return openBlock(), createElementBlock("div", {
					key: `spinner-bar-${bar}`,
					class: "sonner-loading-bar"
				});
			}), 128))])], 8, _hoisted_1$5);
		};
	}
});

//#endregion
//#region src/packages/assets/Loader.vue
var Loader_default = Loader_vue_vue_type_script_setup_true_lang_default;

//#endregion
//#region src/packages/assets/SuccessIcon.vue
const _sfc_main$3 = {};
const _hoisted_1$4 = {
	xmlns: "http://www.w3.org/2000/svg",
	viewBox: "0 0 20 20",
	fill: "currentColor",
	height: "20",
	width: "20"
};
function _sfc_render$3(_ctx, _cache) {
	return openBlock(), createElementBlock("svg", _hoisted_1$4, _cache[0] || (_cache[0] = [createElementVNode("path", {
		"fill-rule": "evenodd",
		d: "M10 18a8 8 0 100-16 8 8 0 000 16zm3.857-9.809a.75.75 0 00-1.214-.882l-3.483 4.79-1.88-1.88a.75.75 0 10-1.06 1.061l2.5 2.5a.75.75 0 001.137-.089l4-5.5z",
		"clip-rule": "evenodd"
	}, null, -1)]));
}
var SuccessIcon_default = /* @__PURE__ */ export_helper_default(_sfc_main$3, [["render", _sfc_render$3]]);

//#endregion
//#region src/packages/assets/InfoIcon.vue
const _sfc_main$2 = {};
const _hoisted_1$3 = {
	xmlns: "http://www.w3.org/2000/svg",
	viewBox: "0 0 20 20",
	fill: "currentColor",
	height: "20",
	width: "20"
};
function _sfc_render$2(_ctx, _cache) {
	return openBlock(), createElementBlock("svg", _hoisted_1$3, _cache[0] || (_cache[0] = [createElementVNode("path", {
		"fill-rule": "evenodd",
		d: "M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a.75.75 0 000 1.5h.253a.25.25 0 01.244.304l-.459 2.066A1.75 1.75 0 0010.747 15H11a.75.75 0 000-1.5h-.253a.25.25 0 01-.244-.304l.459-2.066A1.75 1.75 0 009.253 9H9z",
		"clip-rule": "evenodd"
	}, null, -1)]));
}
var InfoIcon_default = /* @__PURE__ */ export_helper_default(_sfc_main$2, [["render", _sfc_render$2]]);

//#endregion
//#region src/packages/assets/WarningIcon.vue
const _sfc_main$1 = {};
const _hoisted_1$2 = {
	xmlns: "http://www.w3.org/2000/svg",
	viewBox: "0 0 24 24",
	fill: "currentColor",
	height: "20",
	width: "20"
};
function _sfc_render$1(_ctx, _cache) {
	return openBlock(), createElementBlock("svg", _hoisted_1$2, _cache[0] || (_cache[0] = [createElementVNode("path", {
		"fill-rule": "evenodd",
		d: "M9.401 3.003c1.155-2 4.043-2 5.197 0l7.355 12.748c1.154 2-.29 4.5-2.599 4.5H4.645c-2.309 0-3.752-2.5-2.598-4.5L9.4 3.003zM12 8.25a.75.75 0 01.75.75v3.75a.75.75 0 01-1.5 0V9a.75.75 0 01.75-.75zm0 8.25a.75.75 0 100-1.5.75.75 0 000 1.5z",
		"clip-rule": "evenodd"
	}, null, -1)]));
}
var WarningIcon_default = /* @__PURE__ */ export_helper_default(_sfc_main$1, [["render", _sfc_render$1]]);

//#endregion
//#region src/packages/assets/ErrorIcon.vue
const _sfc_main = {};
const _hoisted_1$1 = {
	xmlns: "http://www.w3.org/2000/svg",
	viewBox: "0 0 20 20",
	fill: "currentColor",
	height: "20",
	width: "20"
};
function _sfc_render(_ctx, _cache) {
	return openBlock(), createElementBlock("svg", _hoisted_1$1, _cache[0] || (_cache[0] = [createElementVNode("path", {
		"fill-rule": "evenodd",
		d: "M18 10a8 8 0 11-16 0 8 8 0 0116 0zm-8-5a.75.75 0 01.75.75v4.5a.75.75 0 01-1.5 0v-4.5A.75.75 0 0110 5zm0 10a1 1 0 100-2 1 1 0 000 2z",
		"clip-rule": "evenodd"
	}, null, -1)]));
}
var ErrorIcon_default = /* @__PURE__ */ export_helper_default(_sfc_main, [["render", _sfc_render]]);

//#endregion
//#region src/packages/Toaster.vue?vue&type=script&setup=true&lang.ts
const _hoisted_1 = ["aria-label"];
const _hoisted_2 = [
	"data-sonner-theme",
	"dir",
	"data-theme",
	"data-rich-colors",
	"data-y-position",
	"data-x-position"
];
const isClient = typeof window !== "undefined" && typeof document !== "undefined";
function getDocumentDirection() {
	if (typeof window === "undefined") return "ltr";
	if (typeof document === "undefined") return "ltr";
	const dirAttribute = document.documentElement.getAttribute("dir");
	if (dirAttribute === "auto" || !dirAttribute) return window.getComputedStyle(document.documentElement).direction;
	return dirAttribute;
}
var Toaster_vue_vue_type_script_setup_true_lang_default = /* @__PURE__ */ defineComponent({
	name: "Toaster",
	inheritAttrs: false,
	__name: "Toaster",
	props: {
		id: {},
		invert: {
			type: Boolean,
			default: false
		},
		theme: { default: "light" },
		position: { default: "bottom-right" },
		closeButtonPosition: { default: "top-left" },
		hotkey: { default: () => ["altKey", "KeyT"] },
		richColors: {
			type: Boolean,
			default: false
		},
		expand: {
			type: Boolean,
			default: false
		},
		duration: {},
		gap: { default: GAP },
		visibleToasts: { default: VISIBLE_TOASTS_AMOUNT },
		closeButton: {
			type: Boolean,
			default: false
		},
		toastOptions: { default: () => ({}) },
		class: { default: "" },
		style: {},
		offset: { default: VIEWPORT_OFFSET },
		mobileOffset: { default: MOBILE_VIEWPORT_OFFSET },
		dir: { default: "auto" },
		swipeDirections: {},
		icons: {},
		containerAriaLabel: { default: "Notifications" }
	},
	setup(__props) {
		const props = __props;
		const attrs = useAttrs();
		const toasts = ref([]);
		const filteredToastsById = computed(() => {
			if (props.id) return toasts.value.filter((toast$1) => toast$1.toasterId === props.id);
			return toasts.value.filter((toast$1) => !toast$1.toasterId);
		});
		function filteredToasts(pos, index) {
			return filteredToastsById.value.filter((toast$1) => !toast$1.position && index === 0 || toast$1.position === pos);
		}
		const possiblePositions = computed(() => {
			const posList = filteredToastsById.value.filter((toast$1) => toast$1.position).map((toast$1) => toast$1.position);
			return posList.length > 0 ? Array.from(new Set([props.position].concat(posList))) : [props.position];
		});
		const toastsByPosition = computed(() => {
			const result = {};
			possiblePositions.value.forEach((pos) => {
				result[pos] = toasts.value.filter((t) => t.position === pos);
			});
			return result;
		});
		const heights = ref([]);
		const expanded = ref({});
		const interacting = ref(false);
		watchEffect(() => {
			possiblePositions.value.forEach((pos) => {
				if (!(pos in expanded.value)) expanded.value[pos] = false;
			});
		});
		const actualTheme = ref(props.theme !== "system" ? props.theme : typeof window !== "undefined" ? window.matchMedia && window.matchMedia("(prefers-color-scheme: dark)").matches ? "dark" : "light" : "light");
		const listRef = ref(null);
		const lastFocusedElementRef = ref(null);
		const isFocusWithinRef = ref(false);
		const hotkeyLabel = props.hotkey.join("+").replace(/Key/g, "").replace(/Digit/g, "");
		function removeToast(toastToRemove) {
			if (!toasts.value.find((toast$1) => toast$1.id === toastToRemove.id)?.delete) ToastState.dismiss(toastToRemove.id);
			toasts.value = toasts.value.filter(({ id }) => id !== toastToRemove.id);
			setTimeout(() => {
				if (!toasts.value.find((t) => t.id === toastToRemove.id)) heights.value = heights.value.filter((h) => h.toastId !== toastToRemove.id);
			}, TIME_BEFORE_UNMOUNT + 50);
		}
		function onBlur(event) {
			if (isFocusWithinRef.value && !event.currentTarget?.contains?.(event.relatedTarget)) {
				isFocusWithinRef.value = false;
				if (lastFocusedElementRef.value) {
					lastFocusedElementRef.value.focus({ preventScroll: true });
					lastFocusedElementRef.value = null;
				}
			}
		}
		function onFocus(event) {
			const isNotDismissible = event.target instanceof HTMLElement && event.target.dataset.dismissible === "false";
			if (isNotDismissible) return;
			if (!isFocusWithinRef.value) {
				isFocusWithinRef.value = true;
				lastFocusedElementRef.value = event.relatedTarget;
			}
		}
		function onPointerDown(event) {
			if (event.target) {
				const isNotDismissible = event.target instanceof HTMLElement && event.target.dataset.dismissible === "false";
				if (isNotDismissible) return;
			}
			interacting.value = true;
		}
		watchEffect((onInvalidate) => {
			const unsubscribe = ToastState.subscribe((toast$1) => {
				if (toast$1.dismiss) {
					requestAnimationFrame(() => {
						toasts.value = toasts.value.map((t) => t.id === toast$1.id ? {
							...t,
							delete: true
						} : t);
					});
					return;
				}
				nextTick(() => {
					const indexOfExistingToast = toasts.value.findIndex((t) => t.id === toast$1.id);
					if (indexOfExistingToast !== -1) toasts.value = [
						...toasts.value.slice(0, indexOfExistingToast),
						{
							...toasts.value[indexOfExistingToast],
							...toast$1
						},
						...toasts.value.slice(indexOfExistingToast + 1)
					];
					else toasts.value = [toast$1, ...toasts.value];
				});
			});
			onInvalidate(unsubscribe);
		});
		watchEffect((onInvalidate) => {
			if (typeof window === "undefined") return;
			/**
			* If the theme prop is explicitly set (e.g., 'light' or 'dark'),
			* use it directly and stop watching for system preference.
			*/
			if (props.theme !== "system") {
				actualTheme.value = props.theme;
				return;
			}
			/**
			* Handle "system" theme:
			* Watch the user's OS-level color scheme preference and
			* apply 'dark' or 'light' accordingly.
			*/
			const darkMediaQuery = window.matchMedia("(prefers-color-scheme: dark)");
			/**
			* Helper function to update the actualTheme value
			* based on current media query match.
			*
			* @param {boolean} matches - true if dark mode is preferred
			*/
			const updateTheme = (matches) => {
				actualTheme.value = matches ? "dark" : "light";
			};
			updateTheme(darkMediaQuery.matches);
			/**
			* Media query listener for changes to system preference
			* Compatible with modern browsers and legacy Safari.
			*/
			const handler = (event) => {
				updateTheme(event.matches);
			};
			try {
				darkMediaQuery.addEventListener("change", handler);
			} catch {
				darkMediaQuery.addListener(handler);
			}
			onInvalidate(() => {
				try {
					darkMediaQuery.removeEventListener("change", handler);
				} catch {
					darkMediaQuery.removeListener(handler);
				}
			});
		});
		watchEffect(() => {
			if (listRef.value && lastFocusedElementRef.value) {
				lastFocusedElementRef.value.focus({ preventScroll: true });
				lastFocusedElementRef.value = null;
				isFocusWithinRef.value = false;
			}
		});
		watchEffect(() => {
			if (toasts.value.length <= 1) Object.keys(expanded.value).forEach((pos) => {
				expanded.value[pos] = false;
			});
		});
		watchEffect((onInvalidate) => {
			function handleKeyDown(event) {
				const isHotkeyPressed = props.hotkey.every((key) => event[key] || event.code === key);
				const listRefItem = Array.isArray(listRef.value) ? listRef.value[0] : listRef.value;
				if (isHotkeyPressed) {
					possiblePositions.value.forEach((pos) => {
						expanded.value[pos] = true;
					});
					listRefItem?.focus();
				}
				const isItemActive = document.activeElement === listRef.value || listRefItem?.contains(document.activeElement);
				if (event.code === "Escape" && isItemActive) possiblePositions.value.forEach((pos) => {
					expanded.value[pos] = false;
				});
			}
			if (!isClient) return;
			document.addEventListener("keydown", handleKeyDown);
			onInvalidate(() => {
				document.removeEventListener("keydown", handleKeyDown);
			});
		});
		function handleMouseEnter(event) {
			const target = event.currentTarget;
			const position = target.getAttribute("data-y-position") + "-" + target.getAttribute("data-x-position");
			expanded.value[position] = true;
		}
		function handleMouseLeave(event) {
			if (!interacting.value) {
				const target = event.currentTarget;
				const position = target.getAttribute("data-y-position") + "-" + target.getAttribute("data-x-position");
				expanded.value[position] = false;
			}
		}
		function handleDragEnd() {
			Object.keys(expanded.value).forEach((pos) => {
				expanded.value[pos] = false;
			});
		}
		function handlePointerUp() {
			interacting.value = false;
		}
		function updateHeights(h) {
			heights.value = h;
		}
		function updateHeight(h) {
			const index = heights.value.findIndex((item) => item.toastId === h.toastId);
			if (index !== -1) heights.value[index] = h;
			else {
				const samePositionIndex = heights.value.findIndex((item) => item.position === h.position);
				if (samePositionIndex !== -1) heights.value.splice(samePositionIndex, 0, h);
				else heights.value.unshift(h);
			}
		}
		return (_ctx, _cache) => {
			return openBlock(), createElementBlock(Fragment, null, [createCommentVNode(" Remove item from normal navigation flow, only available via hotkey "), createElementVNode("section", {
				"aria-label": `${_ctx.containerAriaLabel} ${unref(hotkeyLabel)}`,
				tabIndex: -1,
				"aria-live": "polite",
				"aria-relevant": "additions text",
				"aria-atomic": "false"
			}, [(openBlock(true), createElementBlock(Fragment, null, renderList(possiblePositions.value, (pos, index) => {
				return openBlock(), createElementBlock("ol", mergeProps({
					key: pos,
					ref_for: true,
					ref_key: "listRef",
					ref: listRef,
					"data-sonner-toaster": "",
					"data-sonner-theme": actualTheme.value,
					class: props.class,
					dir: _ctx.dir === "auto" ? getDocumentDirection() : _ctx.dir,
					tabIndex: -1,
					"data-theme": _ctx.theme,
					"data-rich-colors": _ctx.richColors,
					"data-y-position": pos.split("-")[0],
					"data-x-position": pos.split("-")[1],
					style: {
						"--front-toast-height": `${heights.value[0]?.height || 0}px`,
						"--width": `${unref(TOAST_WIDTH)}px`,
						"--gap": `${_ctx.gap}px`,
						..._ctx.style,
						...unref(attrs).style,
						...unref(assignOffset)(_ctx.offset, _ctx.mobileOffset)
					}
				}, { ref_for: true }, _ctx.$attrs, {
					onBlur,
					onFocus,
					onMouseenter: handleMouseEnter,
					onMousemove: handleMouseEnter,
					onMouseleave: handleMouseLeave,
					onDragend: handleDragEnd,
					onPointerdown: onPointerDown,
					onPointerup: handlePointerUp
				}), [(openBlock(true), createElementBlock(Fragment, null, renderList(filteredToasts(pos, index), (toast$1, idx) => {
					return openBlock(), createBlock(Toast_default, {
						key: toast$1.id,
						heights: heights.value,
						icons: _ctx.icons,
						index: idx,
						toast: toast$1,
						defaultRichColors: _ctx.richColors,
						duration: _ctx.toastOptions?.duration ?? _ctx.duration,
						class: normalizeClass(_ctx.toastOptions?.class ?? ""),
						descriptionClass: _ctx.toastOptions?.descriptionClass,
						invert: _ctx.invert,
						visibleToasts: _ctx.visibleToasts,
						closeButton: _ctx.toastOptions?.closeButton ?? _ctx.closeButton,
						interacting: interacting.value,
						position: pos,
						closeButtonPosition: _ctx.toastOptions?.closeButtonPosition ?? _ctx.closeButtonPosition,
						style: normalizeStyle(_ctx.toastOptions?.style),
						unstyled: _ctx.toastOptions?.unstyled,
						classes: _ctx.toastOptions?.classes,
						cancelButtonStyle: _ctx.toastOptions?.cancelButtonStyle,
						actionButtonStyle: _ctx.toastOptions?.actionButtonStyle,
						"close-button-aria-label": _ctx.toastOptions?.closeButtonAriaLabel,
						toasts: toastsByPosition.value[pos],
						expandByDefault: _ctx.expand,
						gap: _ctx.gap,
						expanded: expanded.value[pos] || false,
						swipeDirections: props.swipeDirections,
						"onUpdate:heights": updateHeights,
						"onUpdate:height": updateHeight,
						onRemoveToast: removeToast
					}, {
						"close-icon": withCtx(() => [renderSlot(_ctx.$slots, "close-icon", {}, () => [createVNode(CloseIcon_default)])]),
						"loading-icon": withCtx(() => [renderSlot(_ctx.$slots, "loading-icon", {}, () => [createVNode(Loader_default, { visible: toast$1.type === "loading" }, null, 8, ["visible"])])]),
						"success-icon": withCtx(() => [renderSlot(_ctx.$slots, "success-icon", {}, () => [createVNode(SuccessIcon_default)])]),
						"error-icon": withCtx(() => [renderSlot(_ctx.$slots, "error-icon", {}, () => [createVNode(ErrorIcon_default)])]),
						"warning-icon": withCtx(() => [renderSlot(_ctx.$slots, "warning-icon", {}, () => [createVNode(WarningIcon_default)])]),
						"info-icon": withCtx(() => [renderSlot(_ctx.$slots, "info-icon", {}, () => [createVNode(InfoIcon_default)])]),
						_: 2
					}, 1032, [
						"heights",
						"icons",
						"index",
						"toast",
						"defaultRichColors",
						"duration",
						"class",
						"descriptionClass",
						"invert",
						"visibleToasts",
						"closeButton",
						"interacting",
						"position",
						"closeButtonPosition",
						"style",
						"unstyled",
						"classes",
						"cancelButtonStyle",
						"actionButtonStyle",
						"close-button-aria-label",
						"toasts",
						"expandByDefault",
						"gap",
						"expanded",
						"swipeDirections"
					]);
				}), 128))], 16, _hoisted_2);
			}), 128))], 8, _hoisted_1)], 2112);
		};
	}
});

//#endregion
//#region src/packages/Toaster.vue
var Toaster_default = Toaster_vue_vue_type_script_setup_true_lang_default;

//#endregion
//#region src/packages/index.ts
const plugin = { install(app) {
	app.component("Toaster", Toaster_default);
} };
var packages_default = plugin;

//#endregion
export { Toaster_default as Toaster, packages_default as default, toast, useVueSonner };