import { Powerlevel, type SetValueAPIOptions, type SetValueResult } from "@zwave-js/cc";
import { CommandClasses, Duration, type DurationLike, type MaybeNotKnown, type NodeUpdatePayload, type QuerySecurityClasses, type SendCommandOptions, type TranslatedValueID, type ValueDB, type ValueID } from "@zwave-js/core";
import { type BytesView, TypedEventTarget } from "@zwave-js/shared";
import type { Driver } from "../driver/Driver.js";
import type { StatisticsEventCallbacksWithSelf } from "../driver/Statistics.js";
import { DeviceClass } from "./DeviceClass.js";
import type { NodeDump } from "./Dump.js";
import { type NodeStatistics, NodeStatisticsHost } from "./NodeStatistics.js";
import { type DateAndTime, type LifelineHealthCheckResult, type LifelineHealthCheckSummary, type LinkReliabilityCheckOptions, type LinkReliabilityCheckResult, type RefreshInfoOptions, type RouteHealthCheckResult, type RouteHealthCheckSummary, type ZWaveNodeEventCallbacks, type ZWaveNotificationCapability } from "./_Types.js";
import { ZWaveNodeMixins } from "./mixins/index.js";
type AllNodeEvents = ZWaveNodeEventCallbacks & StatisticsEventCallbacksWithSelf<ZWaveNode, NodeStatistics>;
export interface ZWaveNode extends TypedEventTarget<AllNodeEvents>, NodeStatisticsHost {
}
/**
 * A ZWaveNode represents a node in a Z-Wave network. It is also an instance
 * of its root endpoint (index 0)
 */
export declare class ZWaveNode extends ZWaveNodeMixins implements QuerySecurityClasses {
    constructor(id: number, driver: Driver, deviceClass?: DeviceClass, supportedCCs?: CommandClasses[], controlledCCs?: CommandClasses[], valueDB?: ValueDB);
    /**
     * Cleans up all resources used by this node
     */
    destroy(): void;
    /**
     * The device specific key (DSK) of this node in binary format.
     * This is only set if included with Security S2.
     */
    get dsk(): BytesView | undefined;
    /**
     * The user-defined name of this node. Uses the value reported by `Node Naming and Location CC` if it exists.
     *
     * **Note:** Setting this value only updates the name locally. To permanently change the name of the node, use
     * the `commandClasses` API.
     */
    get name(): MaybeNotKnown<string>;
    set name(value: string | undefined);
    /**
     * The user-defined location of this node. Uses the value reported by `Node Naming and Location CC` if it exists.
     *
     * **Note:** Setting this value only updates the location locally. To permanently change the location of the node, use
     * the `commandClasses` API.
     */
    get location(): MaybeNotKnown<string>;
    set location(value: string | undefined);
    /** Whether a SUC return route was configured for this node */
    get hasSUCReturnRoute(): boolean;
    set hasSUCReturnRoute(value: boolean);
    /** The last time a message was received from this node */
    get lastSeen(): MaybeNotKnown<Date>;
    /**
     * The default volume level to be used for activating a Sound Switch.
     * Can be overridden by command-specific options.
     */
    get defaultVolume(): number | undefined;
    set defaultVolume(value: number | undefined);
    /**
     * The default transition duration to be used for transitions like dimming lights or activating scenes.
     * Can be overridden by command-specific options.
     */
    get defaultTransitionDuration(): string | undefined;
    set defaultTransitionDuration(value: string | Duration | DurationLike | undefined);
    /** Returns a list of all value names that are defined on all endpoints of this node */
    getDefinedValueIDs(): TranslatedValueID[];
    /**
     * Updates a value for a given property of a given CommandClass on the node.
     * This will communicate with the node!
     */
    setValue(valueId: ValueID, value: unknown, options?: SetValueAPIOptions): Promise<SetValueResult>;
    /**
     * Requests a value for a given property of a given CommandClass by polling the node.
     * **Warning:** Some value IDs share a command, so make sure not to blindly call this for every property
     */
    pollValue<T = unknown>(valueId: ValueID, sendCommandOptions?: SendCommandOptions): Promise<MaybeNotKnown<T>>;
    /** Returns a list of all `"notification"` event arguments that are known to be supported by this node */
    getSupportedNotificationEvents(): ZWaveNotificationCapability[];
    private _interviewAttempts;
    /** How many attempts to interview this node have already been made */
    get interviewAttempts(): number;
    private _hasEmittedNoS2NetworkKeyError;
    private _hasEmittedNoS0NetworkKeyError;
    /**
     * Starts or resumes a deferred initial interview of this node.
     *
     * **WARNING:** This is only allowed when the initial interview was deferred using the
     * `interview.disableOnNodeAdded` option. Otherwise, this method will throw an error.
     *
     * **NOTE:** It is advised to NOT await this method as it can take a very long time (minutes to hours)!
     */
    interview(): Promise<void>;
    private _refreshInfoPending;
    /**
     * Resets all information about this node and forces a fresh interview.
     * **Note:** This does nothing for the controller node.
     *
     * **WARNING:** Take care NOT to call this method when the node is already being interviewed.
     * Otherwise the node information may become inconsistent.
     */
    refreshInfo(options?: RefreshInfoOptions): Promise<void>;
    /** Updates this node's interview stage and saves to cache when appropriate */
    private setInterviewStage;
    /** Step #1 of the node interview */
    protected queryProtocolInfo(): Promise<void>;
    /**
     * Pings the node to see if it responds
     * @param tryReallyHard Whether the controller should resort to route resolution
     * and explorer frames if the communication fails. Setting this option to `true`
     * can result in multi-second delays.
     */
    ping(tryReallyHard?: boolean): Promise<boolean>;
    /**
     * Step #5 of the node interview
     * Request node info
     */
    protected interviewNodeInfo(): Promise<void>;
    requestNodeInfo(): Promise<NodeUpdatePayload>;
    /** Step #? of the node interview */
    protected interviewCCs(): Promise<boolean>;
    /**
     * Rediscovers all capabilities of a single CC on this node and all endpoints.
     * This can be considered a more targeted variant of `refreshInfo`.
     *
     * WARNING: It is not recommended to await this method!
     */
    interviewCC(cc: CommandClasses): Promise<void>;
    /**
     * Refreshes all non-static values of a single CC from this node (all endpoints).
     * WARNING: It is not recommended to await this method!
     */
    refreshCCValues(cc: CommandClasses): Promise<void>;
    /**
     * Refreshes all non-static values from this node's actuator and sensor CCs.
     * WARNING: It is not recommended to await this method!
     */
    refreshValues(): Promise<void>;
    /**
     * Uses the `commandClasses` compat flag defined in the node's config file to
     * override the reported command classes.
     * @param endpointIndex If given, limits the application of the compat flag to the given endpoint.
     */
    private applyCommandClassesCompatFlag;
    /**
     * Updates the supported CCs of the given endpoint depending on compat flags
     * and certification requirements
     */
    private modifySupportedCCBeforeInterview;
    /** Overwrites the reported configuration with information from a config file */
    protected overwriteConfig(): Promise<void>;
    private centralSceneHandlerStore;
    private clockHandlerStore;
    private hailHandlerStore;
    private notificationHandlerStore;
    private wakeUpHandlerStore;
    private entryControlHandlerStore;
    /**
     * Manually resets a single notification value to idle.
     */
    manuallyIdleNotificationValue(valueId: ValueID): void;
    manuallyIdleNotificationValue(notificationType: number, prevValue: number, endpointIndex?: number): void;
    /**
     * Instructs the node to send powerlevel test frames to the other node using the given powerlevel. Returns how many frames were acknowledged during the test.
     *
     * **Note:** Depending on the number of test frames, this may take a while
     */
    testPowerlevel(testNodeId: number, powerlevel: Powerlevel, healthCheckTestFrameCount: number, onProgress?: (acknowledged: number, total: number) => void): Promise<number>;
    private _healthCheckInProgress;
    /**
     * Returns whether a health check is currently in progress for this node
     */
    isHealthCheckInProgress(): boolean;
    private _healthCheckAborted;
    private _abortHealthCheckPromise;
    /**
     * Aborts an ongoing health check if one is currently in progress.
     *
     * **Note:** The health check may take a few seconds to actually be aborted.
     * When it is, the promise returned by {@link checkLifelineHealth} or
     * {@link checkRouteHealth} will be resolved with the results obtained so far.
     */
    abortHealthCheck(): void;
    /**
     * Checks the health of connection between the controller and this node and returns the results.
     */
    checkLifelineHealth(rounds?: number, onProgress?: (round: number, totalRounds: number, lastRating: number, lastResult: LifelineHealthCheckResult) => void): Promise<LifelineHealthCheckSummary>;
    private checkLifelineHealthInternal;
    /**
     * Checks the health of connection between this node and the target node and returns the results.
     */
    checkRouteHealth(targetNodeId: number, rounds?: number, onProgress?: (round: number, totalRounds: number, lastRating: number, lastResult: RouteHealthCheckResult) => void): Promise<RouteHealthCheckSummary>;
    private checkRouteHealthInternal;
    private _linkReliabilityCheckInProgress;
    /**
     * Returns whether a link reliability check is currently in progress for this node
     */
    isLinkReliabilityCheckInProgress(): boolean;
    private _linkReliabilityCheckAborted;
    private _abortLinkReliabilityCheckPromise;
    /**
     * Aborts an ongoing link reliability check if one is currently in progress.
     *
     * **Note:** The link reliability check may take a few seconds to actually be aborted.
     * When it is, the promise returned by {@link checkLinkReliability} will be resolved with the results obtained so far.
     */
    abortLinkReliabilityCheck(): void;
    /**
     * Tests the reliability of the link between the controller and this node and returns the results.
     */
    checkLinkReliability(options: LinkReliabilityCheckOptions): Promise<LinkReliabilityCheckResult>;
    private checkLinkReliabilityBasicSetOnOff;
    /**
     * Sets the current date, time and timezone (or a subset of those) on the node using one or more of the respective CCs.
     * Returns whether the operation was successful.
     */
    setDateAndTime(now?: Date): Promise<boolean>;
    /**
     * Returns the current date, time and timezone (or a subset of those) on the node using one or more of the respective CCs.
     */
    getDateAndTime(): Promise<DateAndTime>;
    sendResetLocallyNotification(): Promise<void>;
    /** Returns a dump of this node's information for debugging purposes */
    createDump(): NodeDump;
    protected _emit<TEvent extends keyof AllNodeEvents>(event: TEvent, ...args: Parameters<AllNodeEvents[TEvent]>): boolean;
    protected _on<TEvent extends keyof AllNodeEvents>(event: TEvent, callback: AllNodeEvents[TEvent]): this;
    protected _once<TEvent extends keyof AllNodeEvents>(event: TEvent, callback: AllNodeEvents[TEvent]): this;
}
export {};
//# sourceMappingURL=Node.d.ts.map